# hdgl_native_kernel.py
import numpy as np
import struct

class HDGL_VM_32:
    def __init__(self):
        self.D = np.zeros(32)
    def step(self, steps=20):
        for _ in range(steps):
            self.D = np.tanh(self.D + 0.618)

def generate_hdgl_elf(vm, filename="hdgl_kernel.elf"):
    # ----------------------------
    # Minimal x86_64 Linux program:
    # write(1, message, len) + exit(0)
    # ----------------------------
    message = b"Hello from HDGL Linux!\n"
    msg_len = len(message)

    # x86_64 syscall code (sys_write + sys_exit)
    # We'll lattice-seed some NOPs for flavor
    code = b""
    # NOP sled seeded by lattice
    for val in vm.D:
        b = int((val + 1) * 127.5) & 0xFF
        code += struct.pack("B", b)
    code += b"\x90" * 16  # additional NOP padding

    # Syscall instructions
    # mov rax,1 ; mov rdi,1 ; lea rsi,[rip+msg] ; mov rdx,msg_len ; syscall
    code += b"\x48\xc7\xc0\x01\x00\x00\x00"   # mov rax,1
    code += b"\x48\xc7\xc7\x01\x00\x00\x00"   # mov rdi,1
    code += b"\x48\x8d\x35\x12\x00\x00\x00"   # lea rsi,[rip+msg]
    code += struct.pack("<Q", msg_len)[:4]     # mov rdx,msg_len (partial)
    code += b"\x0f\x05"                        # syscall
    # exit(0)
    code += b"\x48\xc7\xc0\x3c\x00\x00\x00"   # mov rax,60
    code += b"\x48\x31\xff"                    # xor rdi,rdi
    code += b"\x0f\x05"                        # syscall
    # message appended
    code += message

    # ----------------------------
    # ELF header (minimal x86_64)
    # ----------------------------
    e_ident = b"\x7fELF" + b"\x02\x01\x01" + b"\x00"*9
    e_type = struct.pack("<H", 2)
    e_machine = struct.pack("<H", 0x3E)
    e_version = struct.pack("<I", 1)
    e_entry = struct.pack("<Q", 0x400078)      # entry point: after headers
    e_phoff = struct.pack("<Q", 64)
    e_shoff = struct.pack("<Q", 0)
    e_flags = struct.pack("<I", 0)
    e_ehsize = struct.pack("<H", 64)
    e_phentsize = struct.pack("<H", 56)
    e_phnum = struct.pack("<H", 1)
    e_shentsize = struct.pack("<H", 0)
    e_shnum = struct.pack("<H", 0)
    e_shstrndx = struct.pack("<H", 0)
    elf_header = e_ident + e_type + e_machine + e_version + e_entry + e_phoff + e_shoff + \
                 e_flags + e_ehsize + e_phentsize + e_phnum + e_shentsize + e_shnum + e_shstrndx

    # Program header
    p_type = struct.pack("<I", 1)  # PT_LOAD
    p_flags = struct.pack("<I", 5) # RX
    p_offset = struct.pack("<Q", 0x78)  # after header
    p_vaddr = struct.pack("<Q", 0x400000)
    p_paddr = struct.pack("<Q", 0x400000)
    p_filesz = struct.pack("<Q", len(code))
    p_memsz = struct.pack("<Q", len(code))
    p_align = struct.pack("<Q", 0x1000)
    prog_header = p_type + p_flags + p_offset + p_vaddr + p_paddr + p_filesz + p_memsz + p_align

    # Write ELF
    with open(filename, "wb") as f:
        f.write(elf_header)
        f.write(prog_header)
        f.write(code)

    print(f"[+] HDGL-native Linux ELF generated: {filename}")
    print("[+] Boot with QEMU:")
    print(f"qemu-system-x86_64 -kernel {filename} -nographic")

def hdgl_native_boot():
    vm = HDGL_VM_32()
    vm.step(steps=20)
    generate_hdgl_elf(vm)

if __name__ == "__main__":
    hdgl_native_boot()
